#----------------------------------------------------------------------------
# Copyright (c) 1998-2008 Autodesk, Inc.
# All rights reserved.
#
# These coded instructions, statements, and computer programs contain
# unpublished proprietary information written by Autodesk, Inc., and are
# protected by Federal copyright law. They may not be disclosed to third
# parties or copied or duplicated in any form, in whole or in part, without
# the prior written consent of Autodesk, Inc.
#----------------------------------------------------------------------------
# Author: Nicolas Desjardins
# Date: 2008-03-26
#----------------------------------------------------------------------------

# Command-line runner for MxsUnit unit tests.
# Requires ruby.
#
# Usage: mxsunit.rb <target program command line> -t <test script patterns>
#
# Example:
#
# mxsunit.rb ProcessHarness.exe ..\3dswin\src\exe\3dsmax.exe -yc -silent -t ..\3dswin\src\dll\DaylightSimulation\**\*unittest.ms
#
# Run from the mxsunit directory, this will execute max under the ProcessHarness
# with network licensing and in silent mode.  The runner will apply any unit
# test scripts ending in unittest.ms found in the DaylightSimulation 
# directory or any of its subdirectories. 
#
# For information on the <test script patterns> syntax see the Dir.glob
# documentation: http://www.ruby-doc.org/core/classes/Dir.html#M002347
#
# The runner works by creating a temporary proxy script file, which invokes
# the Maxscript MxsUnitRunner's run_script_list method, passing it the list
# of found scripts. The unit test report is written out to another temporary 
# file, which this runner spits out to the console after max exits.
#
# To catch crashes under automation, look for "MxsUnit completed" as the last
# line of output.  If that line is not present, max crashed.
#
# The ProcessHarness utility program mentioned above serves to kill max if it
# runs for longer than an hour.  This timeout value can be set as a command line
# parameter.  The source code for ProcessHarness is in the Benchmark System 
# framework.    
#
# This script exits with the same exit code as the target program.

require 'rake'
require 'tempfile'

test_marker_index = ARGV.rindex('-t')
fail "\n   Usage: mxsunit.rb <target program command line> -t <test script patterns>" unless test_marker_index

command_line = ARGV[0...test_marker_index]
fail "No command line given" if command_line.empty?
max_path = command_line.grep(/3dsmax|3dsviz/).first
fail "Could not determine path to 3dsmax or 3dsviz from command line" unless max_path

test_patterns = ARGV[test_marker_index+1..-1]
tests = FileList.new(test_patterns)
fail "No test scripts found" if tests.empty?

test_strings = tests.collect { |t| %Q{"#{File.expand_path(t)}"} }
mxs_test_array = "#( #{test_strings.join ', '} )"

output_file = Tempfile.new('mxsunit_output', File.join(File.dirname(max_path), 'scripts'))
output_file.close

COMPLETED_MARKER= "MxsUnit completed"

proxy_text=<<END_PROXY
global mxsunit_output_file = @"#{output_file.path}"
sleep 20
fileIn "MxsUnit\\\\MxsUnit.ms"
g_mxsunit_runner.run_script_list #{mxs_test_array}
g_mxsunit_runner.reporter.output_to_file mxsunit_output_file
stream = openFile mxsunit_output_file mode:"at"
format "#{COMPLETED_MARKER}\\n" to:stream
close stream
quitMax #noprompt
END_PROXY


proxy_file = Tempfile.new('mxsunit_proxy', File.join(File.dirname(max_path), 'scripts'))
proxy_file.puts proxy_text
proxy_file.close

system "#{command_line.join ' '} -U MAXScript #{File.basename(proxy_file.path)}"
exit_code = $?.exitstatus 
test_output_lines = File.open(output_file.path).readlines
test_output_lines.each { |line| puts line }

if test_output_lines.grep(/#{COMPLETED_MARKER}/).empty?
  puts "Error: MxsUnit did not complete"
end

if 0 != exit_code
  puts "Error: MxsUnit exited with a non-zero exit code"  
end
# add a blank line
puts

exit exit_code
